(************************************************************************
 *                                                                      *
 *   Ant Movie Catalog 4.x                                              *
 *   (C) 2000-2011 Antoine Potten, Mickal Vanneufville                 *
 *   http://www.antp.be/software                                        *
 *                                                                      *
 ************************************************************************
 *                                                                      *
 *   This program is free software; you can redistribute it and/or      *
 *   modify it under the terms of the GNU General Public License        *
 *   as published by the Free Software Foundation; either version 2     *
 *   of the License, or (at your option) any later version.             *
 *                                                                      *
 *   This program is distributed in the hope that it will be useful,    *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of     *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the      *
 *   GNU General Public License for more details.                       *
 *                                                                      *
 ************************************************************************)

unit framemovie;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls, StdCtrls,

  AntJvEdit, AntJvSpin, AntJvExControls, AntJvToolEdit,

  movieclass, ComCtrls;

type
  TMovieFrame = class(TFrame)
    EVideoFormat: TComboBox;
    ETranslatedTitle: TEdit;
    ESubtitles: TComboBox;
    ESource: TComboBox;
    ESize: TEdit;
    EResolution: TEdit;
    EProducer: TEdit;
    EOriginalTitle: TEdit;
    EMediaType: TComboBox;
    EMedia: TEdit;
    ELanguages: TComboBox;
    EFramerate: TComboBox;
    EDirector: TEdit;
    EDescription: TMemo;
    EDate: TDateTimePicker;
    ECountry: TComboBox;
    EComments: TMemo;
    ECategory: TComboBox;
    EBorrower: TComboBox;
    EAudioFormat: TComboBox;
    EActors: TMemo;
    ERating: TAntJvSpinEdit;
    EYear: TAntJvSpinEdit;
    EVideoBitrate: TAntJvSpinEdit;
    ELength: TAntJvSpinEdit;
    EDisks: TAntJvSpinEdit;
    EAudioBitrate: TAntJvSpinEdit;
    EURL: TAntJvComboEditXP;
    LYear: TStaticText;
    LVideoKbps: TStaticText;
    LVideoFormat: TStaticText;
    LURL: TStaticText;
    LTranslatedTitle: TStaticText;
    LSubtitles: TStaticText;
    LSource: TStaticText;
    LSizeUnit: TStaticText;
    LSize: TStaticText;
    LResolution: TStaticText;
    LRating10: TStaticText;
    LRating: TStaticText;
    LProducer: TStaticText;
    LOriginalTitle: TStaticText;
    LMediaType: TStaticText;
    LMedia: TStaticText;
    LLengthMin: TStaticText;
    LLength: TStaticText;
    LLanguages: TStaticText;
    LFramerateFPS: TStaticText;
    LFramerate: TStaticText;
    LDisks: TStaticText;
    LDirector: TStaticText;
    LDescription: TStaticText;
    LDate: TStaticText;
    LCountry: TStaticText;
    LComments: TStaticText;
    LCategory: TStaticText;
    LBorrower: TStaticText;
    LAudioKbps: TStaticText;
    LAudioFormat: TStaticText;
    LActors: TStaticText;

    procedure FrameResize(Sender: TObject);
    procedure FieldChange(Sender: TObject);
    procedure FieldKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure FieldKeyUp(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure FielURLButtonClick(Sender: TObject);
  private
    FModified: Boolean;
    FUpdateCount: Integer;
    FOnFieldChange: TNotifyEvent;
    FOnUserAskListUpdate: TNotifyEvent;
    FOnURLButtonClick: TNotifyEvent;
    FOnURLEnter: TNotifyEvent;

  public
    procedure SaveToObject(AMovie: TMovie; const ForceSave: Boolean = False);
    procedure LoadFromObject(AMovie: TMovie);
    property Modified: Boolean read FModified write FModified;
    property OnFieldChange: TNotifyEvent read FOnFieldChange write FOnFieldChange;
    procedure BeginUpdate;
    procedure EndUpdate;
    function IsUpdating: Boolean;

    property OnUserAskListUpdate: TNotifyEvent read FOnUserAskListUpdate write FOnUserAskListUpdate;
    property OnURLButtonClick: TNotifyEvent read FOnURLButtonClick write FOnURLButtonClick;
    property OnURLEnter: TNotifyEvent read FOnURLEnter write FOnURLEnter;
  end;

implementation

uses
  Global, functions_files;

{$R *.dfm}

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrame.LoadFromObject(AMovie: TMovie);
begin
  BeginUpdate;
  try
    with AMovie do
    begin
      if iDate = 0 then
        EDate.DateTime := Date
      else
        EDate.DateTime := iDate;
      EDate.Checked := iDate <> 0;
      EBorrower.Text := strBorrower;
      EMedia.Text := strMedia;
      EMediaType.Text := strMediaType;
      ESource.Text := strSource;
      ECountry.Text := strCountry;
      ECategory.Text := strCategory;
      EVideoFormat.Text := strVideoFormat;
      EOriginalTitle.Text := strOriginalTitle;
      ETranslatedTitle.Text := strTranslatedTitle;
      EDirector.Text := strDirector;
      EProducer.Text := strProducer;
      ECountry.Text := strCountry;
      EYear.Value := iYear;
      ELength.Value := iLength;
      EActors.Text := strActors;
      EURL.Text := strURL;
      EDescription.Text := strDescription;
      EComments.Text := strComments;
      ESize.Text := strSize;
      EResolution.Text := strResolution;
      ELanguages.Text := strLanguages;
      ESubtitles.Text := strSubtitles;
      if iRating = -1 then
        ERating.Text := ''
      else
        ERating.Value := iRating / 10;
      EAudioFormat.Text := strAudioFormat;
      EFramerate.Text := strFramerate;
      EVideoBitrate.Value := iVideoBitrate;
      EAudioBitrate.Value := iAudioBitrate;
      EDisks.Value := iDisks;
    end;
  finally
    EndUpdate;
  end;
  FModified := False;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrame.SaveToObject(AMovie: TMovie; const ForceSave: Boolean = False);
begin
  if Modified or ForceSave then
    with AMovie do
    begin
      if EDate.Checked then
        iDate := Trunc(EDate.DateTime)
      else
        iDate := 0;
      strOriginalTitle := EOriginalTitle.Text;
      strTranslatedTitle := ETranslatedTitle.Text;
      strDirector := EDirector.Text;
      strProducer := EProducer.Text;
      strCategory := ECategory.Text;
      strCountry := ECountry.Text;
      iYear := EYear.AsInteger;
      iLength := ELength.AsInteger;
      strActors := EActors.Text;
      strURL := EURL.Text;
      strDescription := EDescription.Text;
      strComments := EComments.Text;
      strSize := ESize.Text;
      strVideoFormat := EVideoFormat.Text;
      strResolution := EResolution.Text;
      strLanguages := ELanguages.Text;
      strSubtitles := ESubtitles.Text;
      if ERating.Text = '' then
        iRating := -1
      else
        iRating := Trunc(ERating.Value * 10);
      strMedia := EMedia.Text;
      strMediaType := EMediaType.Text;
      strSource := ESource.Text;
      strAudioFormat := EAudioFormat.Text;
      strFramerate := EFramerate.Text;
      iVideoBitrate := EVideoBitrate.AsInteger;
      iAudioBitrate := EAudioBitrate.AsInteger;
      iDisks := EDisks.AsInteger;
      strBorrower := EBorrower.Text;
    end;
  if not ForceSave then
    FModified := False;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrame.FrameResize(Sender: TObject);
var
  h: Integer;
begin
  h := EVideoFormat.Top - EDescription.Top - 6;
  EDescription.Height := h div 2;
  EComments.Top := EDescription.Top + EDescription.Height + 3;
  EComments.Height := EDescription.Height + (h mod 2);
  LComments.Top := EComments.Top + 4;

  // Fix BUG : Remove auto SelectAll when ComboBox is resized
  EMediaType.SelLength := 0;
  ESource.SelLength := 0;
  EBorrower.SelLength := 0;
  ECountry.SelLength := 0;
  ECategory.SelLength := 0;
  EVideoFormat.SelLength := 0;
  EAudioFormat.SelLength := 0;
  ELanguages.SelLength := 0;
  ESubtitles.SelLength := 0;
  EFramerate.SelLength := 0;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrame.BeginUpdate;
begin
  Inc(FUpdateCount);
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrame.EndUpdate;
begin
  Dec(FUpdateCount);
  if FUpdateCount < 0 then
    FUpdateCount := 0;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

function TMovieFrame.IsUpdating: Boolean;
begin
  Result := FUpdateCount > 0;
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrame.FieldChange(Sender: TObject);
begin
  if not IsUpdating then
  begin
    FModified := True;
    if Assigned(FOnFieldChange) then
      FOnFieldChange(Sender);
  end;
end;

procedure TMovieFrame.FieldKeyDown(Sender: TObject;
  var Key: Word; Shift: TShiftState);
begin
  if (Key = VK_ESCAPE) or ((Key = VK_RETURN) and (Sender <> EDescription) and
    (Sender <> EComments)) then
    if Assigned(OnUserAskListUpdate) then
      OnUserAskListUpdate(Sender);
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrame.FieldKeyUp(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if (Key = Ord('A')) and (Shift = [ssCtrl]) then
  begin
    if Sender is TAntJvSpinEdit then
      TAntJvSpinEdit(Sender).SelectAll
    else if Sender is TEdit then
      TEdit(Sender).SelectAll
    else if Sender is TMemo then
      TMemo(Sender).SelectAll
    else if Sender is TComboBox then
      TComboBox(Sender).SelectAll
    else if Sender is TAntJvComboEditXP then
      TAntJvComboEditXP(Sender).SelectAll;
  end
  else if (Key = VK_RETURN) and (Sender is TAntJvComboEditXP) and Assigned(OnURLEnter) then
    OnURLEnter(Sender);
end;

{-------------------------------------------------------------------------------
-------------------------------------------------------------------------------}

procedure TMovieFrame.FielURLButtonClick(Sender: TObject);
begin
  if Assigned(OnURLButtonClick) then
    OnURLButtonClick(Sender);
end;

end.
